<?php

namespace VM\ApiBundle\Controller;

use FOS\RestBundle\Controller\Annotations\View;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use VM\ApiBundle\Exceptions\ApiRequestException;

class ApiController extends Controller
{
    /**
     * @var
     */
    protected $ApiRequest;

    /**
     * @var string
     */
    protected $realMethodName = '';


    protected $anonymousApiMethods = array();


    /**
     * @View()
     * @return array
     */
    public function indexAction()
    {
        try
        {
            $this->__parseRequest();
            $this->__isAllowedMethod();
            return $this->__execute();
        }
        catch(\Exception $Exception)
        {
            return $this->__createExceptionResponse($Exception);
        }
    }


    /**
     * @return mixed
     * @throws \VM\ApiBundle\Exceptions\ApiRequestException
     */
    protected function __parseRequest()
    {
        if($this->getRequest()->getContentType() == 'json')
        {
            return $this->ApiRequest = json_decode($this->getRequest()->getContent(), true);
        }
        else if($this->getRequest()->getContentType() == 'xml')
        {
            throw new ApiRequestException('Request Content-Type: application/xml not supported yet', 10001);
        }
        else
        {
            throw new ApiRequestException('Request wrong Content-Type. Get application/' . $this->getRequest()->getContentType() . ' nedded application/json or application/xml', 10002);
        }
    }


    /**
     * Check if method is allowed
     *
     * @throws \VM\ApiBundle\Exceptions\ApiRequestException
     */
    protected function __isAllowedMethod()
    {
        if(!in_array($this->ApiRequest['method'], $this->anonymousApiMethods))
        {
            if(!$this->__isServiceAllowedMethod())
            {
                throw new ApiRequestException('Method is not allowed', 10007);
            }
        }
    }


    /**
     * Check if method is assigned to service
     *
     * @return bool
     * @throws \VM\ApiBundle\Exceptions\ApiRequestException
     */
    protected function __isServiceAllowedMethod()
    {
        if(empty($this->ApiRequest['appKey']))
        {
            throw new ApiRequestException('No application key', 10008);
        }

        $Service = $this->__getServiceByAppKey($this->ApiRequest['appKey']);
        $Method = $this->__getMethodBySymbol($this->ApiRequest['method']);

        if(!is_null($Service->getIp()) && ($Service->getIp() != $this->getRequest()->getClientIp()))
        {
            throw new ApiRequestException('Access from non authorization IP address', 10010);
        }


        if(!$Service->getMethods()->contains($Method))
        {
            throw new ApiRequestException('Method is not allowed for service', 10012);
        }

        return true;
    }


    /**
     * @param $appKey
     * @return \VM\ApiBundle\Entity\Service
     * @throws \VM\ApiBundle\Exceptions\ApiRequestException
     */
    protected function __getServiceByAppKey($appKey)
    {
        $Service = $this->getDoctrine()->getRepository('VMApiBundle:Service')->findOneBy(array('appKey' => $appKey));
        if(empty($Service))
        {
            throw new ApiRequestException('Service not found', 10009);
        }
        return $Service;
    }


    /**
     * @param $method
     * @return \VM\ApiBundle\Entity\Method
     * @throws \VM\ApiBundle\Exceptions\ApiRequestException
     */
    protected function __getMethodBySymbol($method)
    {
        $Method = $this->getDoctrine()->getRepository('VMApiBundle:Method')->findOneBySymbol($method);
        if(empty($Method))
        {
            throw new ApiRequestException('Method is not found in allowed method list', 10011);
        }
        return $Method;
    }

    /**
     * @return mixed
     * @throws \VM\ApiBundle\Exceptions\ApiRequestException
     */
    protected function __execute()
    {
        $Service = $this->getService();

        $ThisClass = new \ReflectionClass(get_class($Service));
        $Method = $ThisClass->getMethod($this->realMethodName);

        if(!$Method->isPublic())
        {
            throw new ApiRequestException('Method is not public', 10003);
        }

        $numberOfArgs = $Method->getNumberOfParameters();
        $numberOfRequiredArgs = $Method->getNumberOfRequiredParameters();
        $numberOfSendArguments = (isset($this->ApiRequest['params']) && is_array($this->ApiRequest['params']) ? count($this->ApiRequest['params']) : 0);
        if($numberOfSendArguments < $numberOfRequiredArgs || $numberOfSendArguments > $numberOfArgs)
        {
            throw new ApiRequestException('Wrong number of params', 10004);
        }

        $parameters = (!empty($this->ApiRequest['params'])) ? $this->returnCorrectArrayOfParameters($Method, $this->ApiRequest['params']) : array();
        return $Method->invokeArgs($Service, $parameters);
    }


    /**
     * @param \Exception $Exception
     * @return array
     */
    private function __createExceptionResponse(\Exception $Exception)
    {
        $response = array(
            'error' => array(
                'message' => $Exception->getMessage(),
                'code' => (get_class($Exception) == 'ReflectionException') ? 10013 : $Exception->getCode()
            )
        );

        if($response['error']['code'] == 20003)
        {
            $response['error']['message'] = $Exception->getMessage();
        }


        return $response;
    }


    /**
     * @param $Method
     * @param $parameters
     * @return array
     */
    private function returnCorrectArrayOfParameters($Method, $parameters)
    {
        $array = array();
        $methodParameters = $Method->getParameters();

        if(count($methodParameters) != 0)
        {
            foreach($methodParameters as $ReflectionParameter)
            {
                if(isset($parameters[$ReflectionParameter->getName()]))
                {
                    $array[$ReflectionParameter->getName()] = $parameters[$ReflectionParameter->getName()];
                }
            }
        }

        return $array;
    }


    /**
     * @return object
     * @throws \VM\ApiBundle\Exceptions\ApiRequestException
     */
    private function getService()
    {
        $found = preg_match('/^([^_]*)_(.*)/', $this->ApiRequest['method'], $res);
        if(!$found)
        {
            throw new ApiRequestException('Wrong method', 10005);
        }
        else
        {
            $this->realMethodName = $res[2];
        }

        switch($res[1])
        {
            case 'user':
                return $this->get('api.user');
                break;
            default:
                throw new ApiRequestException('Service not exist', 10006);
                break;
        }
    }
}
